<?php
require_once 'config.php';

if (!isLoggedIn()) {
    header('Location: login.php');
    exit;
}

// Handle delete action
if (isset($_GET['delete'])) {
    $videoId = $_GET['delete'];
    
    // Get video info
    $stmt = $pdo->prepare("SELECT * FROM videos WHERE video_id = ?");
    $stmt->execute([$videoId]);
    $video = $stmt->fetch();
    
    if ($video) {
        // Delete file
        $filePath = UPLOAD_DIR . $video['filename'];
        if (file_exists($filePath)) {
            unlink($filePath);
        }
        
        // Delete from database
        $stmt = $pdo->prepare("DELETE FROM videos WHERE video_id = ?");
        $stmt->execute([$videoId]);
        
        $_SESSION['success_message'] = 'Video deleted successfully';
    }
    
    header('Location: admin.php');
    exit;
}

// Pagination
$page = isset($_GET['page']) ? max(1, intval($_GET['page'])) : 1;
$perPage = 20;
$offset = ($page - 1) * $perPage;

// Search
$search = $_GET['search'] ?? '';
$searchQuery = '';
$searchParams = [];

if (!empty($search)) {
    $searchQuery = " WHERE original_name LIKE ? OR video_id LIKE ?";
    $searchParams = ['%' . $search . '%', '%' . $search . '%'];
}

// Get total count
$stmt = $pdo->prepare("SELECT COUNT(*) FROM videos" . $searchQuery);
$stmt->execute($searchParams);
$totalVideos = $stmt->fetchColumn();
$totalPages = ceil($totalVideos / $perPage);

// Get videos
$stmt = $pdo->prepare("SELECT * FROM videos" . $searchQuery . " ORDER BY uploaded_at DESC LIMIT ? OFFSET ?");
$params = array_merge($searchParams, [$perPage, $offset]);
$stmt->execute($params);
$videos = $stmt->fetchAll();

// Calculate stats
$stmt = $pdo->query("SELECT 
    COUNT(*) as total_videos,
    SUM(file_size) as total_size,
    SUM(views) as total_views
    FROM videos");
$stats = $stmt->fetch();
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Admin Dashboard - Videy</title>
    <script src="https://cdn.tailwindcss.com"></script>
</head>
<body class="bg-gray-50">
    <!-- Header -->
    <header class="bg-white border-b border-gray-200 px-6 py-4 sticky top-0 z-10">
        <div class="flex items-center justify-between max-w-7xl mx-auto">
            <div class="flex items-center gap-4">
                <a href="index.php" class="text-2xl font-bold hover:text-gray-700">videy</a>
                <span class="px-3 py-1 bg-black text-white text-xs rounded-full">Admin</span>
            </div>
            <div class="flex items-center gap-4">
                <span class="text-sm text-gray-600">Welcome, <?php echo htmlspecialchars($_SESSION['admin_username']); ?></span>
                <a href="logout.php" class="px-4 py-2 bg-red-100 text-red-700 hover:bg-red-200 rounded-lg text-sm font-medium transition-colors">
                    Logout
                </a>
            </div>
        </div>
    </header>

    <main class="max-w-7xl mx-auto px-6 py-8">
        <!-- Success Message -->
        <?php if (isset($_SESSION['success_message'])): ?>
            <div class="mb-6 p-4 bg-green-50 border border-green-200 text-green-700 rounded-lg flex items-center justify-between">
                <span><?php echo $_SESSION['success_message']; ?></span>
                <button onclick="this.parentElement.remove()" class="text-green-700 hover:text-green-900">✕</button>
            </div>
            <?php unset($_SESSION['success_message']); ?>
        <?php endif; ?>

        <!-- Stats Cards -->
        <div class="grid grid-cols-1 md:grid-cols-3 gap-6 mb-8">
            <div class="bg-white rounded-xl p-6 shadow-sm border border-gray-200">
                <div class="flex items-center justify-between">
                    <div>
                        <p class="text-sm text-gray-600 mb-1">Total Videos</p>
                        <p class="text-3xl font-bold"><?php echo number_format($stats['total_videos']); ?></p>
                    </div>
                    <div class="w-12 h-12 bg-blue-100 rounded-full flex items-center justify-center">
                        <svg width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" class="text-blue-600">
                            <polygon points="23 7 16 12 23 17 23 7"></polygon>
                            <rect x="1" y="5" width="15" height="14" rx="2" ry="2"></rect>
                        </svg>
                    </div>
                </div>
            </div>

            <div class="bg-white rounded-xl p-6 shadow-sm border border-gray-200">
                <div class="flex items-center justify-between">
                    <div>
                        <p class="text-sm text-gray-600 mb-1">Total Storage</p>
                        <p class="text-3xl font-bold"><?php echo formatBytes($stats['total_size']); ?></p>
                    </div>
                    <div class="w-12 h-12 bg-purple-100 rounded-full flex items-center justify-center">
                        <svg width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" class="text-purple-600">
                            <path d="M21 16V8a2 2 0 0 0-1-1.73l-7-4a2 2 0 0 0-2 0l-7 4A2 2 0 0 0 3 8v8a2 2 0 0 0 1 1.73l7 4a2 2 0 0 0 2 0l7-4A2 2 0 0 0 21 16z"></path>
                        </svg>
                    </div>
                </div>
            </div>

            <div class="bg-white rounded-xl p-6 shadow-sm border border-gray-200">
                <div class="flex items-center justify-between">
                    <div>
                        <p class="text-sm text-gray-600 mb-1">Total Views</p>
                        <p class="text-3xl font-bold"><?php echo number_format($stats['total_views']); ?></p>
                    </div>
                    <div class="w-12 h-12 bg-green-100 rounded-full flex items-center justify-center">
                        <svg width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" class="text-green-600">
                            <path d="M1 12s4-8 11-8 11 8 11 8-4 8-11 8-11-8-11-8z"></path>
                            <circle cx="12" cy="12" r="3"></circle>
                        </svg>
                    </div>
                </div>
            </div>
        </div>

        <!-- Search and Filter -->
        <div class="bg-white rounded-xl p-6 shadow-sm border border-gray-200 mb-6">
            <form method="GET" action="" class="flex gap-4">
                <input type="text" name="search" value="<?php echo htmlspecialchars($search); ?>" placeholder="Search by filename or ID..." class="flex-1 px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-black focus:border-transparent">
                <button type="submit" class="px-6 py-2 bg-black text-white rounded-lg hover:bg-gray-800 transition-colors">
                    Search
                </button>
                <?php if (!empty($search)): ?>
                    <a href="admin.php" class="px-6 py-2 bg-gray-100 text-gray-700 rounded-lg hover:bg-gray-200 transition-colors">
                        Clear
                    </a>
                <?php endif; ?>
            </form>
        </div>

        <!-- Videos Table -->
        <div class="bg-white rounded-xl shadow-sm border border-gray-200 overflow-hidden">
            <div class="overflow-x-auto">
                <table class="w-full">
                    <thead class="bg-gray-50 border-b border-gray-200">
                        <tr>
                            <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Video</th>
                            <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">ID</th>
                            <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Size</th>
                            <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Views</th>
                            <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Uploaded</th>
                            <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Actions</th>
                        </tr>
                    </thead>
                    <tbody class="divide-y divide-gray-200">
                        <?php if (empty($videos)): ?>
                            <tr>
                                <td colspan="6" class="px-6 py-12 text-center text-gray-500">
                                    No videos found
                                </td>
                            </tr>
                        <?php else: ?>
                            <?php foreach ($videos as $video): ?>
                                <tr class="hover:bg-gray-50">
                                    <td class="px-6 py-4">
                                        <div class="flex items-center gap-3">
                                            <div class="w-16 h-16 bg-gray-100 rounded-lg overflow-hidden flex-shrink-0">
                                                <video src="<?php echo SITE_URL . '/uploads/' . $video['filename']; ?>" class="w-full h-full object-cover"></video>
                                            </div>
                                            <div class="min-w-0">
                                                <p class="font-medium text-gray-900 truncate"><?php echo htmlspecialchars($video['original_name']); ?></p>
                                                <p class="text-sm text-gray-500"><?php echo $video['mime_type']; ?></p>
                                            </div>
                                        </div>
                                    </td>
                                    <td class="px-6 py-4">
                                        <code class="px-2 py-1 bg-gray-100 rounded text-sm"><?php echo $video['video_id']; ?></code>
                                    </td>
                                    <td class="px-6 py-4 text-sm text-gray-900">
                                        <?php echo formatBytes($video['file_size']); ?>
                                    </td>
                                    <td class="px-6 py-4 text-sm text-gray-900">
                                        <?php echo number_format($video['views']); ?>
                                    </td>
                                    <td class="px-6 py-4 text-sm text-gray-900">
                                        <?php echo date('M d, Y', strtotime($video['uploaded_at'])); ?>
                                        <br>
                                        <span class="text-gray-500"><?php echo date('H:i', strtotime($video['uploaded_at'])); ?></span>
                                    </td>
                                    <td class="px-6 py-4">
                                        <div class="flex items-center gap-2">
                                            <a href="video.php?id=<?php echo $video['video_id']; ?>" target="_blank" class="px-3 py-1 bg-blue-100 text-blue-700 hover:bg-blue-200 rounded text-sm font-medium transition-colors">
                                                View
                                            </a>
                                            <button onclick="copyVideoLink('<?php echo SITE_URL . '/video.php?id=' . $video['video_id']; ?>')" class="px-3 py-1 bg-green-100 text-green-700 hover:bg-green-200 rounded text-sm font-medium transition-colors">
                                                Copy Link
                                            </button>
                                            <a href="?delete=<?php echo $video['video_id']; ?>" onclick="return confirm('Are you sure you want to delete this video? This action cannot be undone.')" class="px-3 py-1 bg-red-100 text-red-700 hover:bg-red-200 rounded text-sm font-medium transition-colors">
                                                Delete
                                            </a>
                                        </div>
                                    </td>
                                </tr>
                            <?php endforeach; ?>
                        <?php endif; ?>
                    </tbody>
                </table>
            </div>

            <!-- Pagination -->
            <?php if ($totalPages > 1): ?>
                <div class="px-6 py-4 border-t border-gray-200 flex items-center justify-between">
                    <p class="text-sm text-gray-600">
                        Showing <?php echo $offset + 1; ?> to <?php echo min($offset + $perPage, $totalVideos); ?> of <?php echo number_format($totalVideos); ?> videos
                    </p>
                    <div class="flex gap-2">
                        <?php if ($page > 1): ?>
                            <a href="?page=<?php echo $page - 1; ?><?php echo !empty($search) ? '&search=' . urlencode($search) : ''; ?>" class="px-4 py-2 bg-gray-100 hover:bg-gray-200 rounded-lg text-sm font-medium transition-colors">
                                Previous
                            </a>
                        <?php endif; ?>

                        <?php
                        $startPage = max(1, $page - 2);
                        $endPage = min($totalPages, $page + 2);
                        
                        for ($i = $startPage; $i <= $endPage; $i++):
                        ?>
                            <a href="?page=<?php echo $i; ?><?php echo !empty($search) ? '&search=' . urlencode($search) : ''; ?>" class="px-4 py-2 <?php echo $i === $page ? 'bg-black text-white' : 'bg-gray-100 hover:bg-gray-200'; ?> rounded-lg text-sm font-medium transition-colors">
                                <?php echo $i; ?>
                            </a>
                        <?php endfor; ?>

                        <?php if ($page < $totalPages): ?>
                            <a href="?page=<?php echo $page + 1; ?><?php echo !empty($search) ? '&search=' . urlencode($search) : ''; ?>" class="px-4 py-2 bg-gray-100 hover:bg-gray-200 rounded-lg text-sm font-medium transition-colors">
                                Next
                            </a>
                        <?php endif; ?>
                    </div>
                </div>
            <?php endif; ?>
        </div>
    </main>
<!-- Di admin.php stats section: -->
<?php
// Clone statistics
$stmt = $pdo->query("
    SELECT COUNT(*) as cloned_videos 
    FROM videos 
    WHERE user_agent = 'Videy Clone System'
");
$cloneStats = $stmt->fetch();
?>

<div class="bg-white rounded-xl p-6 shadow-sm border border-gray-200">
    <div class="flex items-center justify-between">
        <div>
            <p class="text-sm text-gray-600 mb-1">Cloned Videos</p>
            <p class="text-3xl font-bold"><?php echo number_format($cloneStats['cloned_videos']); ?></p>
        </div>
        <div class="w-12 h-12 bg-green-100 rounded-full flex items-center justify-center">
            <svg width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" class="text-green-600">
                <path stroke-linecap="round" stroke-linejoin="round" d="M8 16H6a2 2 0 01-2-2V6a2 2 0 012-2h8a2 2 0 012 2v2m-6 12h8a2 2 0 002-2v-8a2 2 0 00-2-2h-8a2 2 0 00-2 2v8a2 2 0 002 2z"></path>
            </svg>
        </div>
    </div>
</div>
    <script>
        function copyVideoLink(url) {
            navigator.clipboard.writeText(url).then(function() {
                alert('Link copied to clipboard!');
            }, function() {
                prompt('Copy this link:', url);
            });
        }
    </script>
    
</body>
</html>