<?php require_once 'config.php'; ?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Videyt - Free and Simple Video Hosting</title>
    <script src="https://cdn.tailwindcss.com"></script>
    <style>
        @keyframes spin {
            to { transform: rotate(360deg); }
        }
        .animate-spin {
            animation: spin 1s linear infinite;
        }
    </style>
</head>
<body class="bg-gray-50">
    <header class="bg-white border-b border-gray-200 px-6 py-4">
        <div class="flex items-center justify-between max-w-7xl mx-auto">
            <div class="flex items-center gap-4">
                <h1 class="text-2xl font-bold">Videyt</h1>
                <button onclick="document.getElementById('fileInput').click()" class="px-4 py-2 bg-gray-100 hover:bg-gray-200 rounded-full text-sm font-medium transition-colors">
                    Upload
                </button>
            </div>
            <a href="admin.php" class="text-sm text-gray-600 hover:text-gray-900">Admin</a>
        </div>
    </header>

    <main class="max-w-2xl mx-auto px-6 py-20 text-center">
        <h2 class="text-4xl font-bold mb-4">Free and Simple Video Hosting</h2>
        <p class="text-gray-600 mb-8">Get started without an account</p>
        
        <button onclick="document.getElementById('fileInput').click()" class="px-8 py-4 bg-black text-white rounded-full text-lg font-medium hover:bg-gray-800 transition-colors">
            Upload a Video
        </button>
        
        <br/>
 <p class="text-gray-600 mb-8">Select a video form gallery <br/> Max Upload 100 MB</p>
        <input type="file" id="fileInput" accept="video/*" class="hidden">
    </main>

    <footer class="fixed bottom-0 w-full bg-white border-t border-gray-200 py-4 text-center text-sm text-gray-600">
        <p class="mb-2">Copyright © 2025 Manse Videyt.co</p>
        <div class="flex items-center justify-center gap-4">
            <a href="#" class="hover:text-gray-900">Terms of Service</a>
            <a href="#" class="hover:text-gray-900">Report Abuse</a>
        </div>
    </footer>

    <!-- Confirm Modal -->
    <div id="confirmModal" class="hidden fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center z-50 p-4">
        <div class="bg-white rounded-3xl p-8 max-w-md w-full">
            <h3 class="text-2xl font-bold mb-6 text-center">Confirm Video</h3>
            
            <div class="mb-4 rounded-xl overflow-hidden bg-gray-100">
                <video id="previewVideo" class="w-full h-48 object-cover" controls></video>
            </div>
            
            <p id="fileName" class="text-center text-gray-600 mb-2"></p>
            <p id="fileSize" class="text-center text-sm text-gray-500 mb-8"></p>

            <div class="flex gap-4">
                <button onclick="cancelUpload()" class="flex-1 px-6 py-3 bg-gray-100 hover:bg-gray-200 rounded-full font-medium transition-colors">
                    Cancel
                </button>
                <button onclick="startChunkedUpload()" class="flex-1 px-6 py-3 bg-black text-white hover:bg-gray-800 rounded-full font-medium transition-colors">
                    Upload Video
                </button>
            </div>
        </div>
    </div>

    <!-- Uploading Modal -->
    <div id="uploadingModal" class="hidden fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center z-50 p-4">
        <div class="bg-white rounded-3xl p-12 max-w-md w-full text-center">
            <h3 class="text-2xl font-bold mb-4">Uploading...</h3>
            
            <p class="text-sm text-gray-600 mb-6">Please wait, do not close this page</p>
            
            <div class="mb-6">
                <div class="w-16 h-16 border-4 border-gray-200 border-t-black rounded-full animate-spin mx-auto"></div>
            </div>

            <div class="mb-6">
                <div class="w-full bg-gray-200 rounded-full h-3">
                    <div id="progressBar" class="bg-black h-3 rounded-full transition-all" style="width: 0%"></div>
                </div>
                <p id="progressText" class="text-sm text-gray-600 mt-2">0%</p>
                <p id="speedText" class="text-xs text-gray-500 mt-1"></p>
            </div>

            <p class="text-xs text-gray-500 mb-4">
                ⏱️ Large files may take several minutes
            </p>

            <button onclick="cancelUpload()" class="px-8 py-3 bg-gray-100 hover:bg-gray-200 rounded-full font-medium transition-colors">
                Cancel
            </button>
        </div>
    </div>

    <!-- Share Modal -->
    <div id="shareModal" class="hidden fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center z-50 p-4">
        <div class="bg-white rounded-3xl p-8 max-w-md w-full relative">
            <button onclick="closeShareModal()" class="absolute top-6 right-6 text-gray-400 hover:text-gray-600">
                <svg width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                    <line x1="18" y1="6" x2="6" y2="18"></line>
                    <line x1="6" y1="6" x2="18" y2="18"></line>
                </svg>
            </button>

            <h3 class="text-2xl font-bold mb-6 text-center">Share your video</h3>
            
            <div class="mb-6 rounded-xl overflow-hidden bg-gray-100">
                <video id="sharePreviewVideo" class="w-full h-48 object-cover"></video>
            </div>

            <div class="flex gap-3 mb-6">
                <input type="text" id="shareLink" readonly class="flex-1 px-4 py-3 border border-gray-300 rounded-xl text-sm">
                <button onclick="copyLink()" class="px-6 py-3 bg-gray-100 hover:bg-gray-200 rounded-xl font-medium transition-colors">
                    Copy
                </button>
            </div>

            <button onclick="openVideo()" class="w-full px-6 py-4 bg-black text-white hover:bg-gray-800 rounded-xl font-medium transition-colors">
                Open in Browser
            </button>
        </div>
    </div>

    <script>
        let selectedFile = null;
        let uploadedVideoId = null;
        let isUploading = false;
        let currentXHR = null;

        // Format file size
        function formatBytes(bytes) {
            if (bytes === 0) return '0 Bytes';
            const k = 1024;
            const sizes = ['Bytes', 'KB', 'MB', 'GB'];
            const i = Math.floor(Math.log(bytes) / Math.log(k));
            return Math.round(bytes / Math.pow(k, i) * 100) / 100 + ' ' + sizes[i];
        }

        document.getElementById('fileInput').addEventListener('change', function(e) {
            const file = e.target.files[0];
            if (file && file.type.startsWith('video/')) {
                selectedFile = file;
                
                const preview = document.getElementById('previewVideo');
                preview.src = URL.createObjectURL(file);
                
                document.getElementById('fileName').textContent = file.name;
                document.getElementById('fileSize').textContent = 'Size: ' + formatBytes(file.size);
                document.getElementById('confirmModal').classList.remove('hidden');
            } else {
                alert('Please select a valid video file');
            }
        });

        function cancelUpload() {
            if (currentXHR) {
                currentXHR.abort();
            }
            
            document.getElementById('confirmModal').classList.add('hidden');
            document.getElementById('uploadingModal').classList.add('hidden');
            document.getElementById('shareModal').classList.add('hidden');
            
            selectedFile = null;
            isUploading = false;
            document.getElementById('fileInput').value = '';
            
            document.getElementById('progressBar').style.width = '0%';
            document.getElementById('progressText').textContent = '0%';
        }

        // CHUNKED UPLOAD
        function startChunkedUpload() {
            if (!selectedFile || isUploading) return;

            isUploading = true;
            document.getElementById('confirmModal').classList.add('hidden');
            document.getElementById('uploadingModal').classList.remove('hidden');

            const chunkSize = 2 * 1024 * 1024; // 2MB per chunk (lebih kecil = lebih stabil)
            const totalChunks = Math.ceil(selectedFile.size / chunkSize);
            let currentChunk = 0;
            let uploadedBytes = 0;
            const startTime = Date.now();
            
            // Generate unique identifier untuk upload session ini
            const uploadId = 'upload_' + Date.now() + '_' + Math.random().toString(36).substr(2, 9);

            function uploadNextChunk() {
                if (!isUploading) {
                    console.log('Upload cancelled');
                    return;
                }

                const start = currentChunk * chunkSize;
                const end = Math.min(start + chunkSize, selectedFile.size);
                const chunk = selectedFile.slice(start, end);

                const formData = new FormData();
                formData.append('chunk', chunk);
                formData.append('chunkIndex', currentChunk);
                formData.append('totalChunks', totalChunks);
                formData.append('fileName', selectedFile.name);
                formData.append('fileSize', selectedFile.size);
                formData.append('fileType', selectedFile.type);
                formData.append('uploadId', uploadId);

                currentXHR = new XMLHttpRequest();
                currentXHR.timeout = 60000; // 1 minute per chunk

                currentXHR.upload.addEventListener('progress', function(e) {
                    if (e.lengthComputable) {
                        const chunkLoaded = uploadedBytes + e.loaded;
                        const totalProgress = (chunkLoaded / selectedFile.size) * 100;
                        
                        document.getElementById('progressBar').style.width = totalProgress + '%';
                        document.getElementById('progressText').textContent = 
                            Math.round(totalProgress) + '% - Chunk ' + (currentChunk + 1) + '/' + totalChunks;
                        
                        // Calculate speed
                        const elapsed = (Date.now() - startTime) / 1000;
                        const speed = chunkLoaded / elapsed;
                        const remaining = (selectedFile.size - chunkLoaded) / speed;
                        
                        document.getElementById('speedText').textContent = 
                            'Speed: ' + formatBytes(speed) + '/s - ETA: ' + Math.round(remaining) + 's';
                    }
                });

                currentXHR.addEventListener('load', function() {
                    if (currentXHR.status === 200) {
                        try {
                            const response = JSON.parse(currentXHR.responseText);
                            
                            if (response.success) {
                                uploadedBytes += chunk.size;
                                currentChunk++;
                                
                                if (currentChunk < totalChunks) {
                                    // Upload next chunk
                                    setTimeout(uploadNextChunk, 100); // Small delay between chunks
                                } else {
                                    // All chunks uploaded successfully
                                    console.log('Upload complete!', response);
                                    uploadedVideoId = response.video_id;
                                    showShareModal(response.video_url, response.share_url);
                                    isUploading = false;
                                }
                            } else {
                                throw new Error(response.message || 'Upload failed');
                            }
                        } catch (e) {
                            handleUploadError('Error processing response: ' + e.message);
                        }
                    } else {
                        handleUploadError('Server error: HTTP ' + currentXHR.status);
                    }
                });

                currentXHR.addEventListener('error', function() {
                    handleUploadError('Network error. Please check your connection and try again.');
                });

                currentXHR.addEventListener('timeout', function() {
                    handleUploadError('Upload timeout. Please try again or use a smaller file.');
                });

                currentXHR.open('POST', 'upload_chunk.php');
                currentXHR.send(formData);
            }

            // Start uploading chunks
            uploadNextChunk();
        }

        function handleUploadError(message) {
            isUploading = false;
            alert('Upload Error: ' + message);
            cancelUpload();
        }

        function showShareModal(videoUrl, shareUrl) {
            document.getElementById('uploadingModal').classList.add('hidden');
            document.getElementById('shareModal').classList.remove('hidden');
            
            document.getElementById('sharePreviewVideo').src = videoUrl;
            document.getElementById('shareLink').value = shareUrl;
        }

        function closeShareModal() {
            document.getElementById('shareModal').classList.add('hidden');
            document.getElementById('fileInput').value = '';
            selectedFile = null;
        }

        function copyLink() {
            const linkInput = document.getElementById('shareLink');
            linkInput.select();
            linkInput.setSelectionRange(0, 99999);
            
            try {
                document.execCommand('copy');
                alert('Link copied to clipboard!');
            } catch (err) {
                prompt('Copy this link:', linkInput.value);
            }
        }

        function openVideo() {
            const link = document.getElementById('shareLink').value;
            window.location.href = link;
        }
    </script>
</body>
</html>