<?php
require_once 'config.php';

// Set timeout per chunk (lebih pendek karena chunk kecil)
set_time_limit(120);
ini_set('max_execution_time', 120);

header('Content-Type: application/json');

// Log function
function logError($message) {
    $log = date('Y-m-d H:i:s') . " - " . $message . "\n";
    file_put_contents(__DIR__ . '/chunk_upload_log.txt', $log, FILE_APPEND);
}

try {
    // Validate request
    if (!isset($_FILES['chunk']) || $_FILES['chunk']['error'] !== UPLOAD_ERR_OK) {
        throw new Exception('No chunk uploaded or upload error');
    }

    // Get parameters
    $chunkIndex = intval($_POST['chunkIndex'] ?? -1);
    $totalChunks = intval($_POST['totalChunks'] ?? 0);
    $fileName = $_POST['fileName'] ?? '';
    $fileSize = intval($_POST['fileSize'] ?? 0);
    $fileType = $_POST['fileType'] ?? '';
    $uploadId = $_POST['uploadId'] ?? '';

    // Validate parameters
    if ($chunkIndex < 0 || $totalChunks <= 0 || empty($fileName) || empty($uploadId)) {
        throw new Exception('Invalid parameters');
    }

    logError("Receiving chunk $chunkIndex/$totalChunks for $fileName");

    // Create temp directory
    $tempDir = UPLOAD_DIR . 'temp/';
    if (!is_dir($tempDir)) {
        if (!mkdir($tempDir, 0755, true)) {
            throw new Exception('Cannot create temp directory');
        }
    }

    // Safe filename for temp chunks
    $safeTempName = preg_replace('/[^a-zA-Z0-9_-]/', '_', $uploadId);
    $chunkPath = $tempDir . $safeTempName . '.part' . $chunkIndex;

    // Save chunk
    if (!move_uploaded_file($_FILES['chunk']['tmp_name'], $chunkPath)) {
        throw new Exception('Failed to save chunk ' . $chunkIndex);
    }

    logError("Chunk $chunkIndex saved successfully");

    // If this is the last chunk, merge all chunks
    if ($chunkIndex == $totalChunks - 1) {
        logError("Last chunk received, starting merge...");

        // Generate video ID
        $videoId = generateVideoId();
        $attempts = 0;
        while ($attempts < 10) {
            $stmt = $pdo->prepare("SELECT id FROM videos WHERE video_id = ?");
            $stmt->execute([$videoId]);
            if ($stmt->rowCount() === 0) break;
            $videoId = generateVideoId();
            $attempts++;
        }

        if ($attempts >= 10) {
            throw new Exception('Failed to generate unique video ID');
        }

        // Get file extension
        $ext = strtolower(pathinfo($fileName, PATHINFO_EXTENSION));
        if (empty($ext)) {
            $ext = 'mp4'; // default
        }

        $finalFileName = $videoId . '.' . $ext;
        $finalPath = UPLOAD_DIR . $finalFileName;

        // Merge all chunks into final file
        $output = fopen($finalPath, 'wb');
        if (!$output) {
            throw new Exception('Cannot create final file');
        }

        $totalWritten = 0;
        for ($i = 0; $i < $totalChunks; $i++) {
            $chunkFile = $tempDir . $safeTempName . '.part' . $i;
            
            if (!file_exists($chunkFile)) {
                fclose($output);
                unlink($finalPath);
                throw new Exception('Chunk ' . $i . ' not found');
            }

            $input = fopen($chunkFile, 'rb');
            if (!$input) {
                fclose($output);
                unlink($finalPath);
                throw new Exception('Cannot read chunk ' . $i);
            }

            $written = stream_copy_to_stream($input, $output);
            $totalWritten += $written;
            
            fclose($input);
            unlink($chunkFile); // Delete chunk after merging
        }
        
        fclose($output);

        logError("Merge complete. Total size: $totalWritten bytes");

        // Verify file size
        $actualSize = filesize($finalPath);
        if ($actualSize != $fileSize) {
            unlink($finalPath);
            throw new Exception("File size mismatch. Expected: $fileSize, Got: $actualSize");
        }

        // Save to database
        $stmt = $pdo->prepare("
            INSERT INTO videos (video_id, filename, original_name, file_size, mime_type, ip_address, user_agent)
            VALUES (?, ?, ?, ?, ?, ?, ?)
        ");
        
        $stmt->execute([
            $videoId,
            $finalFileName,
            $fileName,
            $fileSize,
            $fileType,
            $_SERVER['REMOTE_ADDR'] ?? 'unknown',
            $_SERVER['HTTP_USER_AGENT'] ?? 'unknown'
        ]);

        logError("Database insert successful for video ID: $videoId");

        // Return success with video info
        echo json_encode([
            'success' => true,
            'video_id' => $videoId,
            'video_url' => SITE_URL . '/uploads/' . $finalFileName,
            'share_url' => SITE_URL . '/video.php?id=' . $videoId,
            'file_size' => formatBytes($fileSize),
            'message' => 'Upload completed successfully'
        ]);

    } else {
        // Chunk saved, waiting for more
        echo json_encode([
            'success' => true,
            'chunk' => $chunkIndex,
            'message' => 'Chunk ' . $chunkIndex . ' uploaded'
        ]);
    }

} catch (Exception $e) {
    logError("ERROR: " . $e->getMessage());
    
    // Clean up on error
    if (isset($finalPath) && file_exists($finalPath)) {
        unlink($finalPath);
    }
    
    echo json_encode([
        'success' => false,
        'message' => $e->getMessage()
    ]);
}
?>