<?php
require_once 'config.php';

header('Content-Type: application/json');

// Check if file was uploaded
if (!isset($_FILES['video']) || $_FILES['video']['error'] !== UPLOAD_ERR_OK) {
    echo json_encode(['success' => false, 'message' => 'No file uploaded or upload error']);
    exit;
}

$file = $_FILES['video'];

// Validate file size
if ($file['size'] > MAX_FILE_SIZE) {
    echo json_encode(['success' => false, 'message' => 'File too large. Max size: ' . formatBytes(MAX_FILE_SIZE)]);
    exit;
}

// Validate file type
if (!in_array($file['type'], ALLOWED_FORMATS)) {
    echo json_encode(['success' => false, 'message' => 'Invalid file type. Only video files allowed.']);
    exit;
}

// Generate unique video ID
$videoId = generateVideoId();
while (true) {
    $stmt = $pdo->prepare("SELECT id FROM videos WHERE video_id = ?");
    $stmt->execute([$videoId]);
    if ($stmt->rowCount() === 0) break;
    $videoId = generateVideoId();
}

// Get file extension
$ext = pathinfo($file['name'], PATHINFO_EXTENSION);
$filename = $videoId . '.' . $ext;

// Create uploads directory if not exists
if (!is_dir(UPLOAD_DIR)) {
    mkdir(UPLOAD_DIR, 0755, true);
}

// Move uploaded file
$destination = UPLOAD_DIR . $filename;
if (!move_uploaded_file($file['tmp_name'], $destination)) {
    echo json_encode(['success' => false, 'message' => 'Failed to save file']);
    exit;
}

// Save to database
try {
    $stmt = $pdo->prepare("
        INSERT INTO videos (video_id, filename, original_name, file_size, mime_type, ip_address, user_agent)
        VALUES (?, ?, ?, ?, ?, ?, ?)
    ");
    
    $stmt->execute([
        $videoId,
        $filename,
        $file['name'],
        $file['size'],
        $file['type'],
        $_SERVER['REMOTE_ADDR'],
        $_SERVER['HTTP_USER_AGENT'] ?? ''
    ]);

    echo json_encode([
        'success' => true,
        'video_id' => $videoId,
        'video_url' => SITE_URL . '/uploads/' . $filename,
        'share_url' => SITE_URL . '/video.php?id=' . $videoId
    ]);

} catch (PDOException $e) {
    // Delete uploaded file if database insert fails
    unlink($destination);
    echo json_encode(['success' => false, 'message' => 'Database error: ' . $e->getMessage()]);
}
?>